--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Quest Goal/Reward Functions
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- These are functions to use as parameters for quest definitions.
-- They create functions that are included in the quest definition structure.
--
-- Called with no parameters, those functions return true or false to indicate
-- whether the player has achieved a particular goal, or as rewards they
-- produce a side-effect.
-- 
-- Called with a non-nil parameter, they return a string describing the
-- goal or reward, and a true/false indicating whether the goal or reward
-- should be displayed to players or is only for development.
-------------------------------------------------------------------------------

local function NullFunction(d)
	if d then return "(ERROR)",false
	else return true
	end
end

-------------------------------------------------------------------------------
-- Quest Goals/Requirements

function MinRank(r)
	local n=tonumber(r)
	return function(d)
		if d then return GetString("q_minrank", GetString("rank_"..tostring(n))), true
		else return (gSim.rank >= n)
		end
	end
end

function MaxRank(r)
	local n=tonumber(r)
	return function(d)
		if d then return GetString("q_maxrank", GetString("rank_"..tostring(n))), false
		else return (gSim.rank <= n)
		end
	end
end

function MinMoney(m)
	local n=tonumber(m)
	return function(d)
		if d then return GetString("q_mincash", bsgDollars(n)), true
		else return (gSim.money >= n)
		end
	end
end

function MaxMoney(m)
	local n=tonumber(m)
	return function(d)
		if d then return GetString("q_maxcash", bsgDollars(n)), true
		else return (gSim.money <= n)
		end
	end
end

function MinWeek(w)
	local n=tonumber(w)
	return function(d)
		if d then return GetString("q_minweek", bsgDate(n,0), tostring(n)), false
		else return (gSim.weeks >= n)
		end
	end
end

function MaxWeek(w)
	local n=tonumber(w)
	return function(d)
		if d then return GetString("q_maxweek", bsgDate(n,0), tostring(n)), false
		else return (gSim.weeks <= n)
		end
	end
end

function RequireItem(name, count)
	local item=LItem:ByName(tostring(name))
	if not item then
		bsgDevWarning("Undefined item in quest definition: "..tostring(name))
		return NullFunction
	else
		local n=count or 1
		n = tonumber(n)
		return function(d)
			if d then
				local name = item.name
				if item.lab and (not item.known) then name="q_item_unknown" end
				
				local more = n - item.inventory
				if more <= 0 then
					return GetString("q_item", tostring(n), GetString(name)), true
				else
					return GetString("q_item_more", tostring(n), GetString(name), tostring(more)), true
				end
			else return (item.inventory >= n)
			end
		end
	end
end

function RequireLabItem(name)
	local item=LItem:ByName(tostring(name))
	if not item then
		bsgDevWarning("Undefined item in quest definition: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then
				return GetString("q_item_lab", GetString(item.name)), true
			else
				-- Only one lab to stock
				local l = LLaboratory._Laboratories[1]
				if l then return l.stock[item.name]
				else return false
				end
			end
		end
	end
end

function RequireRecipe(name)
	local item=LItem:ByName(tostring(name))
	if not item then
		bsgDevWarning("Undefined item in quest definition: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then
				local name = item.name
				if item.lab and (not item.known) then name="q_item_unknown" end
				return GetString("q_recipe", GetString(name)), true
			else return (item.known)
			end
		end
	end
end

function RequireRecipeUnknown(name)
	local item=LItem:ByName(tostring(name))
	if not item then
		bsgDevWarning("Undefined item in quest definition: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_recipenot", GetString(item.name)), true
			else return (not item.known)
			end
		end
	end
end

function RequireRecipesKnown(typeName, knownCount)
	local type = LProductType:ByName(typeName)
	local count = knownCount
	if not type then
		bsgDevWarning("Undefined item type in quest definition: "..tostring(typeName))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_typeknown", GetString(type.name), tostring(count))
			else return (type.knownCount >= count)
			end
		end
	end
end

function RequireTotalRecipesKnown(knownCount)
	local count = knownCount
	return function(d)
		if d then return GetString("q_totalknown", tostring(count))
		else
			local total = 0
			for _,type in pairs(LProductType._ByIndex) do
				total = total + type.knownCount
			end
			return (total >= count)
		end
	end
end

function RequirePort(name)
	local p=LPort:ByName(tostring(name))
	if not p then
		bsgDevWarning("Undefined port in quest definition: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_port", GetString(p.name)), true
			else return (p.available)
			end
		end
	end
end

function RequireBuilding(name)
	local b=LBuilding:ByName(tostring(name))
	if not b then
		bsgDevWarning("Undefined building in quest definition: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_building", GetString(b.name)), true
			else return (b.owned)
			end
		end
	end
end

function RequireQuest(name)
	local qn = tostring(name)
	return function(d)
		if d then return GetString("q_quest", GetString(qn), qn), false
		else
			local q = LQuest:ByName(qn)
			if not q then
				bsgDevWarning("Undefined quest in quest definition: "..qn)
				return true
			else
				return q.complete
			end
		end
	end
end

function RequireQuestIncomplete(name)
	local qn = tostring(name)
	return function(d)
		if d then return GetString("q_noquest", GetString(qn), qn), false
		else
			local q = LQuest:ByName(qn)
			if not q then
				bsgDevWarning("Undefined quest in quest definition: "..qn)
				return true
			else
				return (not q.complete)
			end
		end
	end
end

function RequireWeeks(n)
	local w=tonumber(n)
	return function(d, q)
		if d then return GetString("q_weeks", tostring(w), bsgDate(gSim.questWeek  + w, 0)), true
		else
			if q and gSim.quest == q then gSim.requireWeek = gSim.questWeek + w end
			return (gSim.weeks >= gSim.questWeek  + w)
		end
	end
end

function RequireNotFirstPeek()
	return function(d)
		if d then return "NOT available during First Peek"
		else
			if bsgCheckConfig("firstpeek") then return false
			else return true
			end
		end
	end
end

function RequireMedal(name)
	local medal = tostring(name)
	local ok = false
	for _,m in ipairs(Simulator._Medals) do
		if m.name==medal then ok=true break end
	end
	if not ok then
		bsgDevWarning("Undefined medal in quest definition: "..medal)
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_medal", GetString("med_"..medal)), true
			else return gSim.medals[medal]
			end
		end
	end
end

function RequireNoMedal(name)
	local medal = tostring(name)
	local ok = false
	for _,m in ipairs(Simulator._Medals) do
		if m.name==medal then ok=true break end
	end
	if not ok then
		bsgDevWarning("Undefined medal in quest definition: "..medal)
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_nomedal", GetString("med_"..medal)), true
			else return (gSim.medals[medal] ~= true)
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Quest Gifts/Rewards

function ShowLedger(tab)
	local t = tostring(tab)
	if not (t == "messages" or t == "factories" or t == "inventory") then
		bsgDevWarning("Undefined ledger tab in quest definition: "..t)
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_ledger", t), false
			else
				SetLedgerContents(t)
				Pause(5)
			end
		end
	end
end

function ShowRecipes()
	return function(d)
		if d then return GetString("q_recipes"), false
		else
			PlaySound("reward_4")
			DoModal("ui/recipebook.lua")
			Pause(5)
		end
	end
end

function AwardRank(r)
	local n = tonumber(r)
	return function(d)
		if d then return GetString("q_rank", GetString("rank_"..tostring(n))), false
		else
			gSim.rank = n
			gSim:QueueMessage(GetString("quest_promotion", GetString("rank_"..tostring(n))))
			
			-- FIRSTPEEK: Promotion, time-stamp, weeks, rank
			if fpWrite then fpWrite { "Promotion", gSim.weeks, gSim.rank } end
		end
	end
end

function AwardMessage(msg)
	local m=tostring(msg)
	return function(d)
		if d then return GetString("q_message", m), false
		else gSim:QueueMessage(GetString(m))
		end
	end
end

function AwardMoney(m)
	local n=tonumber(m)
	return function(d)
		if d then return GetString("q_awardcash", bsgDollars(m)), false
		else
			gSim:AdjustMoney(m)
			if m > 0 then gSim:QueueMessage(GetString("quest_money", bsgDollars(m))) end
		end
	end
end

function AwardItem(name,count)
	local item=LItem:ByName(tostring(name))
	if not item then
		bsgDevWarning("Undefined item in quest definition: "..tostring(name))
		return NullFunction
	else
		local n=count or 1
		n = tonumber(n)
		return function(d)
			if d then return GetString("q_awarditem", tostring(n), GetString(item.name)), false
			else
				item.inventory = item.inventory + n
				item.usetime = gSim.weeks
				if n > 0 then
					if item.recipe then gSim:QueueMessage(GetString("quest_product", tostring(n), GetString(item.name)))
					else gSim:QueueMessage(GetString("quest_ingredient", tostring(n), GetString(item.name)))
					end
				end
				if not item.recipe then gSim:InventoryChanged() end
			end
		end
	end
end

function AwardRecipe(name)
	local item=LItem:ByName(tostring(name))
	if not item then
		bsgDevWarning("Undefined item in quest definition: "..tostring(name))
		return NullFunction
	elseif not item.recipe then
		bsgDevWarning("AwardRecipe item has no recipe: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_awardrecipe", GetString(item.name)), false
			else
				if not item.known then item:EnableRecipe("quest_recipe") end
			end
		end
	end
end

function AwardBuilding(name)
	local b=LBuilding:ByName(tostring(name))
	if not b then
		bsgDevWarning("Undefined building in quest definition: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_awardbuilding", GetString(b.name), b.name), false
			else
				b:MarkOwned()
				gSim:QueueMessage(GetString("quest_building", GetString(b.name)))
			end
		end
	end
end

function EnablePort(p)
	local port = LPort:ByName(tostring(p))
	if not port then
		bsgDevWarning("Undefined port in quest definition: "..tostring(p))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_awardport", GetString(port.name)), false
			else
				port.available = true
				gSim:QueueMessage(GetString("quest_port", GetString(port.name)))
			end
		end
	end
end

function DelayQuest(name,weeks)
	local qn = tostring(name)
	local w = tonumber(weeks)
	return function(d)
		if d then return GetString("q_delay", qn, tostring(w)), false
		else
			local q = LQuest:ByName(qn)
			if not q then bsgDevWarning("Undefined quest in quest definition: "..qn)
			else q.eligibleTime = gSim.weeks + w
			end
		end
	end
end

function OfferQuest(name)
	local qn = tostring(name)
	return function(d)
		if d then return GetString("q_offer", qn), false
		else
			local q = LQuest:ByName(qn)
			if not q then bsgDevWarning("Undefined quest in quest definition: "..qn)
			else
				local char = questCharacter or q.starter[1]
				q:Offer(questCharacter)
				Pause(5)
			end
		end
	end
end

function MarkComplete(name)
	local qn = tostring(name)
	return function(d)
		if d then return GetString("q_markcomplete", qn), false
		else
			local q = LQuest:ByName(qn)
			if not q then bsgDevWarning("Undefined quest in quest definition: "..qn)
			else q.complete = true
			end
		end
	end
end

function ShowDialog(msg, char)
	local m = tostring(msg)
	local c = LCharacter:ByName(tostring(char))
	if char and not c then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_dialog", m, tostring(c)), false
			else
				local speaker = c or questCharacter
				local body = GetString(m) or m
				DisplayDialog { "ui/chartalk.lua", char=speaker, body="#"..body, ok="ok" }
				Pause(5)
--				SwapToModal("ui/portview.lua")
			end
		end
	end
end

function AwardLogo()
	return function(d)
		if d then return GetString("q_logo"), false
		else
			DoModal("ui/logoedit_basic.lua")
		end
	end
end

function AwardRefreshPort()
	return function(d)
		if d then return "", false
		else
			SwapToModal("ui/portview.lua")
			StopPulsing()
		end
	end
end

function AwardTimeWarp(n)
	local w = n or 4
	return function(d)
		if d then return "",false
		else
			gSim.days = 0
			gSim.weeks = gSim.weeks + w
			gSim:GetScore()
			UpdateBadge(gSim)
		end
	end
end

function GameOver()
	return function(d)
		if d then return "",false
		else
			-- Force SF to reload, with sign and fireworks
			SwapToModal("ui/portview.lua")
		end
	end
end

-------------------------------------------------------------------------------
-- Emotion settings for Requirements/Goals and Gifts/Rewards

function RequireHappy(char)
	local c = LCharacter:ByName(tostring(char))
	if (not char) or (not c) then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_happy", GetString(c.name))
			else return (c:IsHappy())
			end
		end
	end
end

function RequireNeutral(char)
	local c = LCharacter:ByName(tostring(char))
	if (not char) or (not c) then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_neutral", GetString(c.name))
			else return (c:IsNeutral())
			end
		end
	end
end

function RequireAngry(char)
	local c = LCharacter:ByName(tostring(char))
	if (not char) or (not c) then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_angry", GetString(c.name))
			else return (c:IsAngry())
			end
		end
	end
end

function SetHappy(char)
	local c = LCharacter:ByName(tostring(char))
	if (not char) or (not c) then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_happy", GetString(c.name))
			else
				local target = LCharacter.kHappy + 10
				if c.mood < target then c:SetMood(target) end
			end
		end
	end
end

function SetNeutral(char)
	local c = LCharacter:ByName(tostring(char))
	if (not char) or (not c) then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_neutral", GetString(c.name))
			else c:SetMood(LCharacter.kDefault)
			end
		end
	end
end

function SetAngry(char)
	local c = LCharacter:ByName(tostring(char))
	if (not char) or (not c) then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_angry", GetString(c.name))
			else
				local target = LCharacter.kAngry - 10
				if c.mood > target then c:SetMood(target) end
			end
		end
	end
end

function LowerMood(char, value)
	local value = value or 10
	local c = LCharacter:ByName(tostring(char))
	if (not char) or (not c) then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_lowermood", GetString(c.name), tostring(value))
			else c:SetMood(c.mood - value)
			end
		end
	end
end

function RaiseMood(char, value)
	local value = value or 10
	local c = LCharacter:ByName(tostring(char))
	if (not char) or (not c) then
		bsgDevWarning("Undefined character in quest definition: "..tostring(char))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_raisemood", GetString(c.name), tostring(value))
			else c:SetMood(c.mood + value)
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Bad Thing "awards"

function ClearItem(name)
	local item=LItem:ByName(tostring(name))
	if not item then
		bsgDevWarning("Undefined item in quest definition: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_clearitem", tostring(n), GetString(item.name)), false
			else
				item.inventory = 0
				item.usetime = gSim.weeks
				if not item.recipe then gSim:InventoryChanged() end
			end
		end
	end
end

function StopFactory(name)
	local b=LBuilding:ByName(tostring(name))
	local ok = false
	for f in LFactory:AllFactories() do
		if f == b then ok=true break end
	end
	if not b then
		bsgDevWarning("Undefined building in quest definition: "..tostring(name))
		return NullFunction
	elseif not ok then
		bsgDevWarning("Building in quest definition is not a Factory: "..tostring(name))
		return NullFunction
	else
		return function(d)
			if d then return GetString("q_stopfactory", GetString(b.port.name))
			else
				if b.owned then
					b:SetConfiguration(b.product.name, 0)
					LFactory:ProjectProduction()
					UpdateLedgerContents("factories")
				end
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Quest Variables for Requirements/Goals and Gifts/Rewards

function VarEquals(var, val)
	local v = tostring(var)
	local n = tonumber(val)
	LQuest._Variables[v] = LQuest._Variables[v] or 0
	return function(d)
		if d then return GetString("q_vareq", v, tostring(n)), false
		else return (LQuest:Variable(v) == n)
		end
	end
end

function VarNotEqual(var, val)
	local v = tostring(var)
	local n = tonumber(val)
	LQuest._Variables[v] = LQuest._Variables[v] or 0
	return function(d)
		if d then return GetString("q_varne", v, tostring(n)), false
		else return (LQuest:Variable(v) ~= n)
		end
	end
end

function VarLessThan(var, val)
	local v = tostring(var)
	local n = tonumber(val)
	LQuest._Variables[v] = LQuest._Variables[v] or 0
	return function(d)
		if d then return GetString("q_varlt", v, tostring(n)), false
		else return (LQuest:Variable(v) < n)
		end
	end
end

function VarMoreThan(var, val)
	local v = tostring(var)
	local n = tonumber(val)
	LQuest._Variables[v] = LQuest._Variables[v] or 0
	return function(d)
		if d then return GetString("q_vargt", v, tostring(n)), false
		else return (LQuest:Variable(v) > n)
		end
	end
end

function IncrementVar(var)
	local v = tostring(var)
	LQuest._Variables[v] = LQuest._Variables[v] or 0
	return function(d)
		if d then return GetString("q_varinc", v), false
		else
			LQuest._Variables[v] = LQuest:Variable(v) + 1
		end
	end
end

function DecrementVar(var)
	local v = tostring(var)
	LQuest._Variables[v] = LQuest._Variables[v] or 0
	return function(d)
		if d then return GetString("q_vardec", v), false
		elseif LQuest:Variable(v) > 0 then LQuest._Variables[v] = LQuest:Variable(v) - 1
		end
	end
end

function SetVar(var, val)
	local v = tostring(var)
	local n = tonumber(val)
	LQuest._Variables[v] = LQuest._Variables[v] or 0
	return function(d)
		if d then return GetString("q_varset", v, tostring(n)), false
		else LQuest._Variables[v] = n
		end
	end
end
